#ifndef __ARUCO_MARKER_DETECTOR_HPP__
#define __ARUCO_MARKER_DETECTOR_HPP__

#include <opencv2/highgui.hpp>
#include <opencv2/imgproc.hpp>
#include <opencv2/calib3d.hpp>
#include <opencv2/aruco.hpp>

#include <vector>
#include <iostream>

#include <Eigen/Core>
#include <Eigen/Geometry>
#include <opencv2/core/eigen.hpp>


class ArucoMarkerDetector {
  
public:
  
  ArucoMarkerDetector(int argc, char** argv);
  
  /// @param[out] cTmk 4-by-4 matrix, pose of marker w.r.t. camera
  int detectMarker(const cv::Mat& scene, cv::Mat& cTmk);
  
private:
  
    int dictionaryId;
    bool showRejected;
    bool estimatePose;
    float markerLength;
    
    cv::Ptr<cv::aruco::DetectorParameters> detectorParams;
    
    cv::VideoCapture inputVideo;
    int waitTime;
    
    double totalTime = 0;
    int totalIterations = 0;
    
    cv::Ptr<cv::aruco::Dictionary> dictionary;
    
    cv::Mat camMatrix, distCoeffs;
  
  const char* about = "Basic marker detection";
  const char* keys  =
        "{d        |       | dictionary: DICT_4X4_50=0, DICT_4X4_100=1, DICT_4X4_250=2,"
        "DICT_4X4_1000=3, DICT_5X5_50=4, DICT_5X5_100=5, DICT_5X5_250=6, DICT_5X5_1000=7, "
        "DICT_6X6_50=8, DICT_6X6_100=9, DICT_6X6_250=10, DICT_6X6_1000=11, DICT_7X7_50=12,"
        "DICT_7X7_100=13, DICT_7X7_250=14, DICT_7X7_1000=15, DICT_ARUCO_ORIGINAL = 16}"
        "{v        |       | Input from video file, if ommited, input comes from camera }"
        "{ci       | 0     | Camera id if input doesnt come from video (-v) }"
        "{c        |       | Camera intrinsic parameters. Needed for camera pose }"
        "{l        | 0.1   | Marker side lenght (in meters). Needed for correct scale in camera pose }"
        "{dp       |       | File of marker detector parameters }"
        "{r        |       | show rejected candidates too }";

  /**
  */
  static bool readCameraParameters(std::string filename, cv::Mat &camMatrix, cv::Mat &distCoeffs) {
      cv::FileStorage fs(filename, cv::FileStorage::READ);
      if(!fs.isOpened())
	  return false;
      fs["camera_matrix"] >> camMatrix;
      fs["distortion_coefficients"] >> distCoeffs;
      return true;
  }



  /**
  */
  static bool readDetectorParameters(std::string filename, cv::Ptr<cv::aruco::DetectorParameters> &params) {
      cv::FileStorage fs(filename, cv::FileStorage::READ);
      if(!fs.isOpened())
	  return false;
      fs["adaptiveThreshWinSizeMin"] >> params->adaptiveThreshWinSizeMin;
      fs["adaptiveThreshWinSizeMax"] >> params->adaptiveThreshWinSizeMax;
      fs["adaptiveThreshWinSizeStep"] >> params->adaptiveThreshWinSizeStep;
      fs["adaptiveThreshConstant"] >> params->adaptiveThreshConstant;
      fs["minMarkerPerimeterRate"] >> params->minMarkerPerimeterRate;
      fs["maxMarkerPerimeterRate"] >> params->maxMarkerPerimeterRate;
      fs["polygonalApproxAccuracyRate"] >> params->polygonalApproxAccuracyRate;
      fs["minCornerDistanceRate"] >> params->minCornerDistanceRate;
      fs["minDistanceToBorder"] >> params->minDistanceToBorder;
      fs["minMarkerDistanceRate"] >> params->minMarkerDistanceRate;
      fs["cornerRefinementMethod"] >> params->cornerRefinementMethod;
      fs["cornerRefinementWinSize"] >> params->cornerRefinementWinSize;
      fs["cornerRefinementMaxIterations"] >> params->cornerRefinementMaxIterations;
      fs["cornerRefinementMinAccuracy"] >> params->cornerRefinementMinAccuracy;
      fs["markerBorderBits"] >> params->markerBorderBits;
      fs["perspectiveRemovePixelPerCell"] >> params->perspectiveRemovePixelPerCell;
      fs["perspectiveRemoveIgnoredMarginPerCell"] >> params->perspectiveRemoveIgnoredMarginPerCell;
      fs["maxErroneousBitsInBorderRate"] >> params->maxErroneousBitsInBorderRate;
      fs["minOtsuStdDev"] >> params->minOtsuStdDev;
      fs["errorCorrectionRate"] >> params->errorCorrectionRate;
      return true;
  }

  int run(int argc, char** argv);

  void createTransformMat(const cv::Mat& r, const cv::Mat& t, cv::Mat& T);

};


#endif 